// src/lib/pixel/buildMessage.ts
import { PIXEL_ERRORS, SANDBOX_CASES, isSandbox } from "./codes";

type PixelCodes = {
  code?: string | null;
  uuid?: string | null;
  hash?: string | null;
  approved?: boolean | null;
  reason?: string | null;
};

export type FrontMessage = {
  title: string;
  icon: 'success'|'error'|'warning'|'info';
  html?: string;   // Para SweetAlert2
  text?: string;   // Fallback para alert()
};

function htmlBlock(lines: string[]) {
  return `<div style="text-align:left">${lines.filter(Boolean).map(l => `<p>${l}</p>`).join("")}</div>`;
}

/**
 * PRIORIDAD DE MENSAJES (corregida):
 * 1) Autorización (401/403) → error de permisos/allowlist/keys.
 * 2) Estado de negocio del API (PAID/DECLINED/ERROR) → manda.
 * 3) Códigos del proveedor (Pixel) → detalle adicional.
 * 4) Casos SANDBOX por monto → solo si no hubo estado/códigos.
 * 5) Fallback genérico.
 */
export function buildFrontMessage({
  pixelMsg,
  pixelCodes,
  result,
  amount,
  api,
  providerHttp, // opcional: http del proveedor (sale/3DS), ej. 412, 422...
}: {
  pixelMsg?: string;
  pixelCodes?: PixelCodes;
  result?: any; // payload del widget
  amount?: number;
  api?: { http?: number; status?: string; message?: string }; // BMT /confirm
  providerHttp?: number;
}): FrontMessage {
  const apiHttp = api?.http;
  const apiStatus = (api?.status || '').toUpperCase();
  const raw = (api?.message || pixelMsg || "").toLowerCase();

  // 1) NoAccess (401/403)
  if (apiHttp === 401 || apiHttp === 403 || /unauthorized|no autorizado/.test(raw)) {
    return {
      title: "No autorizado",
      icon: "error",
      html: htmlBlock([
        "<b>Detalle:</b> Revisa X-API-Key / X-App-Key en el servidor.",
        "Asegura que <i>Origin/Referer</i> coincidan EXACTO con la allowlist.",
        api?.message ? `<b>Mensaje:</b> ${api.message}` : ""
      ])
    };
  }

  // 2) Estado de negocio (API BMT)
  if (apiStatus === "PAID") {
    return { title: "Pago acreditado", icon: "success", text: "¡La recarga fue procesada con éxito!" };
  }
  if (apiStatus === "DECLINED" || /declined|declinada/.test(raw)) {
    // Caso especial: proveedor 412 (límite de intentos / precondición)
    if (providerHttp === 412 || /límite de intentos|limite de intentos|precondition/i.test(raw)) {
      return {
        title: "Intentos excedidos",
        icon: "warning",
        html: htmlBlock([
          "<b>Detalle:</b> El límite de intentos para este email fue superado.",
          "Por favor intenta de nuevo en 24 horas o utiliza otro medio/email.",
          api?.message ? `<b>Mensaje:</b> ${api.message}` : ""
        ])
      };
    }
    return {
      title: "Pago declinado",
      icon: "error",
      html: htmlBlock([
        api?.message ? `<b>Mensaje:</b> ${api.message}` : "La transacción fue declinada.",
        pixelCodes?.code ? `<b>Código:</b> ${pixelCodes.code}` : ""
      ])
    };
  }
  if (apiStatus === "ERROR") {
    return {
      title: "Error al procesar",
      icon: "error",
      html: htmlBlock([
        api?.message ? `<b>Mensaje:</b> ${api.message}` : "No se pudo procesar la transacción.",
        pixelCodes?.code ? `<b>Código:</b> ${pixelCodes.code}` : ""
      ])
    };
  }

  // 3) Códigos del proveedor (Pixel)
  if (pixelCodes?.code && PIXEL_ERRORS[pixelCodes.code]) {
    return {
      title: "Pago no aprobado",
      icon: "error",
      html: htmlBlock([
        `<b>Detalle:</b> ${PIXEL_ERRORS[pixelCodes.code]}`,
        pixelMsg ? `<b>Mensaje:</b> ${pixelMsg}` : "",
        `<b>Código:</b> ${pixelCodes.code}`,
        pixelCodes.uuid ? `<b>UUID:</b> ${pixelCodes.uuid}` : ""
      ])
    };
  }

  // 4) Casos SANDBOX (solo si no hay estado/códigos anteriores)
  if (isSandbox() && Number.isFinite(amount!) && SANDBOX_CASES[amount!]) {
    const c = SANDBOX_CASES[amount!];
    return {
      title: c.title,
      icon: c.icon,
      html: htmlBlock([
        pixelMsg ? `<b>Mensaje:</b> ${pixelMsg}` : `Simulación sandbox con amount = ${amount}`,
        pixelCodes?.code ? `<b>Código:</b> ${pixelCodes.code}` : ""
      ])
    };
  }

  // 5) Fallback genérico
  return {
    title: "No se pudo procesar la transacción",
    icon: "error",
    html: htmlBlock([
      api?.message ? `<b>API:</b> ${api.message}` : "",
      pixelMsg ? `<b>Mensaje:</b> ${pixelMsg}` : "",
      pixelCodes?.code ? `<b>Código:</b> ${pixelCodes.code}` : ""
    ])
  };
}
