'use client';

import { useState } from 'react';
import { currentUser } from "@/lib/session/currentUser";
import { initOrderClient, confirmOrderClient } from "@/lib/bmt/client";
import type { InitResponse } from "@/lib/bmt/types";
import { buildFrontMessage } from "@/lib/pixel/buildMessage";

export default function DepositDialog() {
	const [amount, setAmount] = useState<number>(1);
	const [loading, setLoading] = useState(false);
	const [log, setLog] = useState<string>('');

	async function handleDeposit() {
		setLoading(true);
		setLog('');
		try {
			if (!amount || amount < 1) throw new Error('Monto inválido (mínimo 1).');

			const initResp: InitResponse = await initOrderClient({
				amount,
				currency: 'HNL',
				description: 'Recarga Hondubet',
				customer: { name: currentUser.name, email: currentUser.email },
				billing: currentUser.billing,
			});
			if (!initResp?.success) throw new Error('No se pudo crear la orden');
			setLog(l => l + `Init OK: ${initResp.order_id}\n`);

			await new Promise<void>((resolve, reject) => {
				if (!window.BMTCheckout) return reject(new Error('Widget no disponible'));

				window.BMTCheckout.open({
					checkoutUrl: initResp.checkout_url,
					onResult: async (result: any) => {
						try {
							// Provider http hint (si tu widget lo expone; si no, déjalo undefined)
							const providerHttp = result?.http ?? undefined;

							const conf = await confirmOrderClient({
								order_id: initResp.order_id,
								status: result?.status || 'APPROVED',
								payment_uuid: result?.payment_uuid ?? null,
								payment_hash: result?.payment_hash ?? null,
								message: result?.message,
							});
							setLog(l => l + `Confirm: ${JSON.stringify(conf)}\n`);

							const msg = buildFrontMessage({
								pixelMsg: result?.message,
								pixelCodes: result?.pixel_codes, // si vienen del widget
								result,
								amount,
								api: { http: conf?.data?.http, status: conf?.status, message: conf?.message },
								providerHttp,
							});

							if ((window as any).Swal) {
								await (window as any).Swal.fire({ title: msg.title, icon: msg.icon, html: msg.html, text: msg.text });
							} else {
								alert(`${msg.title}${msg.text ? `\n${msg.text}` : ''}`);
							}

							if (conf?.success && (conf?.status === 'PAID' || conf?.status === 'PENDING')) {
								resolve(); // no lances error en PENDING: ya mostraste el modal y el backend guardó estado
							} else {
								reject(new Error(conf?.message || 'Declinada'));
							}
						} catch (e) { reject(e); }
					},
				});
			});

			// SIN alerta adicional aquí: el builder ya mostró el mensaje correcto.
		} catch (err: any) {
			console.error(err);
			if ((window as any).Swal) {
				await (window as any).Swal.fire({ title: 'Error en el proceso', icon: 'error', text: err?.message || 'Error desconocido' });
			} else {
				alert(err?.message || 'Error en el proceso');
			}
		} finally {
			setLoading(false);
		}
	}

	return (
		<div className="space-y-4 border border-slate-800 rounded-xl p-4">
			<div className="space-y-2">
				<label className="block text-sm text-slate-300">Monto (HNL)</label>
				<input
					type="number"
					min={1}
					value={amount}
					onChange={(e) => setAmount(Number(e.target.value))}
					className="w-full bg-slate-900 border border-slate-700 rounded-md px-3 py-2 focus:outline-none"
				/>
			</div>

			<button
				onClick={handleDeposit}
				disabled={loading}
				className="inline-flex items-center gap-2 bg-emerald-600 hover:bg-emerald-500 disabled:opacity-50 text-white rounded-lg px-4 py-2"
			>
				{loading ? 'Procesando…' : 'Depositar'}
			</button>

			{log && (
				<pre className="text-xs bg-black/40 p-3 rounded-md overflow-auto max-h-64">
					{log}
				</pre>
			)}
		</div>
	);
}
